(function() {

    /**
     * Basic map object.
     * @param {string} mapContainer ID of the map's div container.
     * @param {Object} opt_options Optional configuration parameters.
     * @param {string} opt_options.tileUrl Tile url for the netzkarte tiles.
     */
    window.ETRMap = function(mapContainer, options) {
        this.mapContainer = mapContainer;
        this.url = options.tileUrl || 'static/tiles/';
        this.url += (this.url.substr(-1) === '/') ? '' : '/';
        this.type = options.monitorType === 'big' ? 'big' : 'small';

        this.markerUrl = options.markerUrl;
        this.map = new OpenLayers.Map(this.mapContainer, {});
    };

    var ETRMap = window.ETRMap;

    /**
     * Configuration for different monitor sizes.
     */
    ETRMap.prototype.monitorConfig = {
        small: {
            layer: 'zoom1_v1',
            maxResolution: 8991.428571,
            extent: new OpenLayers.Bounds(55275.4396022, 5447393.24226, 1943475.43961, 6571321.8137)
        },
        big: {
            layer: 'zoom1_v2',
            maxResolution: 8000,
            extent: new OpenLayers.Bounds(1259.17148606, 5452123.16036, 2001259.17149, 6566123.16036)
        },
        high: {
            layer: 'zoom2',
            maxResolution: 7296,
            extent: new OpenLayers.Bounds(550120.0, 5468703.0, 1494040.0, 6555123.0)
        }
    };

    /**
     * Load the map and the tms layer by type.
     * @param {String} type Monitor and resolution type.
     */
    ETRMap.prototype.loadMap = function(type) {
        var config = this.monitorConfig[type];

        // clear map
        for (var i = 0; i < this.map.layers.length; i++) {
            this.map.removeLayer(this.map.layers[i]);
        }

        // update options based on the layer type
        this.map.setOptions({
            maxExtent: config.extent,
            numZoomLevels: 6,
            zoom: 1
        });

        // load tile layer
        var tileLayer = new OpenLayers.Layer.TMS(type, this.url, {
            getURL: this.tileUrlFunction,
            type: 'png',
            maxResolution: config.maxResolution,
            maxExtent:  config.extent,
            layername: config.layer
        });

        this.map.addLayer(tileLayer);
    };

    /**
     * OpenLayers tile function for TMS layer.
     * @param {OpenLayers.Bounds} bounds Current map extent.
     * @returns {String} url Tile url.
     */
    ETRMap.prototype.tileUrlFunction = function(bounds) {
        bounds = this.adjustBounds(bounds);
        var res = this.getServerResolution();
        var x = Math.round((bounds.left - this.tileOrigin.lon) / (res * this.tileSize.w));
        var y = Math.round((bounds.bottom - this.tileOrigin.lat) / (res * this.tileSize.h));
        var z = this.getServerZoom();
        var path = this.serviceVersion + '/' + this.layername +
            '/' + z + '/' + x + '/' + y + '.' + this.type;
        var url = this.url;

        if (OpenLayers.Util.isArray(url)) {
            url = this.selectUrl(path, url);
        }

        if (this.map.maxExtent.intersectsBounds(bounds)) {
            return url + path;
        }
    };

    /**
     * Zoom map view to full extent.
     * @param {Number} lon Longitude
     * @param {Number} lat Latitude
     */
    ETRMap.prototype.zoomToFullExtent = function(lon, lat) {
        this.loadMap(this.type);
        var center = this.map.getMaxExtent().getCenterLonLat();
        this.map.setCenter(center, /* zoom */ 2);
        var lonlat = this.convertCoordinates(lon, lat);
        this.addMapMarker(lonlat);
    };

    /**
     * Add marker icon to the map.
     * @param {Number} lon Longitude
     * @param {Number} lat Latitude
     */
    ETRMap.prototype.addMapMarker = function(lonlat) {
        var markerId = 'divMapMarker';
        var oldElem = document.getElementById(markerId);

        if (oldElem && oldElem.parentElement) {
            oldElem.parentElement.removeChild(oldElem);
        }

        // Create a container with the map marker
        var div = document.createElement('div');
        div.id = markerId;
        div.innerHTML = '<img src="' + this.markerUrl + '"/>';
        div.style.position = 'absolute';
        div.style['z-index'] = 1000000000000000; // should be enough

        // positioning
        var pixel = this.map.getPixelFromLonLat(lonlat);
        var x = pixel.x - 13; // halfe image size
        var y = pixel.y - 13; // halfe image size
        div.style.top = y + 'px';
        div.style.left = x + 'px';

        var map = document.getElementById(this.mapContainer);
        map.appendChild(div);
    };

    /**
     * Zoom map map to coordinate.
     * @param {Number} lon Longitude
     * @param {Number} lat Latitude
     */
    ETRMap.prototype.zoomToCoordinates = function(lon, lat) {
        this.loadMap('high');
        var lonlat = this.convertCoordinates(lon, lat);
        this.map.setCenter(lonlat, 5);
        this.addMapMarker(lonlat);
    };

    /**
     * Convert coordinates from wgs84 to the
     *  projection used by the application.
     * @param {Number} lat Latitude
     * @param {Number} lon Longitude
     * @returns {OpenLayers.LonLat} Reprojected coordinates.
     */
    ETRMap.prototype.convertCoordinates = function(lon, lat) {
        try {
            var worldProj = new OpenLayers.Projection('EPSG:4326');
            var swissProj = new OpenLayers.Projection('EPSG:900913');
            var swissCoords = OpenLayers.Projection.transform(
                new OpenLayers.Geometry.Point(lon, lat), worldProj, swissProj);
            return new OpenLayers.LonLat(swissCoords.x, swissCoords.y);
        } catch (e) {
            this.logError('Unable to translate coordinates.');
            this.zoomToFullExtent();
        }
    };

    /**
     * Some error handling.
     * @param {string} msg The error message
     */
    ETRMap.prototype.logError = function(msg) {
        document.getElementById('error').innerHTML = msg;
    };
})();
